%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  Hydrostatic 1st and 2nd order Clairaut equation solver %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Estimates the spherical harmonic degree 2 and 4 of the hydrostatic 
% shape of the Earth, correct to the second order in the deviation to 
% sphericity, by integration of linear differential systems.
%
% The equations used are differential equations that generalise, 
% coorect to 2nd order, the 1st order ones of Clairaut. These can be found 
% in our paper: Chambat, Ricard, Valette, 2010, "Flattening of the Earth:
% further from hydrostaticity than previously estimated", Geophys. J. Int.,
% submitted. It is also, with misprints, in:
% - the book of Lanzano, 1982, ch. 2.02, p. 50-51. 
% - the book of Kopal, 1960, page 70,
% - the paper of Nakiboglu, 1982 
% (complete references in Chambat et al, 2010)
%
% The main script is the present hydrostatic.m. To use it : just run it 
% with Matlab in the directory were other given files are.
% The script uses a data file:
% - prem.dat : a discretized Earth model (see section 'Reads Earth model')
% and some function files :
% - degree2.m : differential system for spherical harmonic 2
% - degree4.m : differential system for spherical harmonic 4
% - func_integrates.m : integrates a function with a Runge-Kutta scheme
% - func_f. m : estimates the function to intergrate with func_integrates.m
%
% At each reference radius the shape of the layer is :
% s(theta,lambda) = r*(1+f2(r)*P2(theta,lambda)+f4(r)*P4(theta,lambda))
% where theta, lambda are colatitude and longitude, P2 and P4 are the
% Legendre polynomia of degree 2 and 4. Note : the integral of 
% Pn^2(cos theta) over the unit sphere is 4*pi/(2*n+1). 
% 'Legendre polynomia degree' is below said simply 'degree'.
% The script evaluates f2 and f4. Results are stored as follows :
% - F(:;1)=f2(r) evaluated at first order ; F(:;2)= r*df2/dr
% - F2: same evaluated correct to the second order
% - F4: same for degree 4
%
%  F. Chambat, ENS de Lyon, France, november 2009 - may 2010
%  frederic.chambat@ens-lyon.fr
%  http://frederic.chambat.free.fr/hydrostatic


disp('-------------------------')
disp('Begining of hydrostatic.m')
close all;
format long; 
clear;F2saved=zeros(1,2); % Comment this line after a run if you want to 
% iterate as done by most authors. It's here only to compare our numerical 
% integration with their because there is no theoretical justification for 
% iteration.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Data and integration parameters               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Earth Data
GM0=3.986000979e14; % Geocentric gravitational constant, without atmosphere (m3 s-2)
R0=6371.230e3;      % Physical radius (m) (real value)
G=6.67428e-11;      % Gravitational constant (m3 kg-1 s-2)
om=7.2921150e-5;    % Earth angular velocity (rad s-1)
IsM=1.342354e13;    % Inertia ratio (inertia/mass) of the Earth (m2)
iratio0=IsM/R0^2;   % I/MR^2 of the Earth (without dimension)
m0=om^2*R0^3/GM0;   % Adimensional centifugal acceleration of the Earth
ae=6378.137e3;      % Scaling length for J2 and J4


% Integration parameters
r0=10;              % Starting radius (m) to begin the integration
                    % Must be smaller than r(2)
options=odeset('RelTol',1e-12,'AbsTol',1e-16); 
                    % Sets relative and absolute tolerance for ode solver
                    % May need to be changed, eg. depending upon resolution
                    % of Earth model etc. 1e-12 and 1e-16 are usually ok.
ifigure=0;          % Set to 1 if you do want figures to appear,
                    % and to 0 if you don't (fastest).
iGM=0;              % Set to 1 to use the R and GM= G * M of your model
                    % and to 0 (recommended) to use the actual R(=R0) and                    
                    % GM(=GM0) of the Earth (see below).

% Benchmark. With PREM and iGM=0 one finds :
%J2first *1e6, 1st order  =  1073.19150619
%J2      *1e6, 2nd order  =  1072.10571513
%J4      *1e6, 2nd order  =  -2.96343371
%(J2-J2first)  *1e6       =  -1.08579106
%J2 best *1e6,            =  1071.20566135

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Reads Earth model                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% If needed, change the model here : provide another file.dat and change 
% the 3 following lines. What is needed are radius and density at
% discrete points within the Earth. At the discontinuities the radius
% must appear twice, with 2 different values of density.
% The indice in tables ranges from 1 at the center of the Earth
% to n at the surface.

%load prem.dat
%r=prem(:,2); % Radius (m) 
%d=prem(:,3); % Density (kg m-3)
% My prefered model (not distributed) :
load    memmo_rho091216.dat
r=memmo_rho091216(:,1);
d=memmo_rho091216(:,2);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Processes Earth model                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

n=length(r);

% Identifies the layers.
% A change in layer is when the radius has twice the same value.
j=1; % Layer number
nstart(j)=1;    % Indice that starts the layer j
for i=2:n-1
    if abs(r(i+1)-r(i))/r(i) < 1e-12 % Radii are equal
        nend(j)=i;  % Indice that ends the layer j
        j=j+1;
        nstart(j)=i+1;
    end
end
nend(j)=n;
J=j; % Total number of layers
if (r0 >= r(2));disp('***** break : r0 > r(2) ! ******');stop;end
    
% Renormalisation to the actual Earth Radius
if iGM==0
    r=r*R0/r(n);
end
% Vector of radii ri in which radius 0 is replaced with a small one r0
ri=r;ri(1)=r0;

% Computes several global quantities ('Sr' = sphere of radius r).
R=r(n); % Physical radius (m)
mass=4*pi*func_integrates(r,d.*r.^2,J,nstart,nend); % mass(r) within Sr
inertia=8*pi/3*func_integrates(r,d.*r.^4,J,nstart,nend);% inertia within Sr
dbar=3*mass./(4*pi*ri.^3);dbar(1)=d(1); % mean density within Sr   
gamma=d./dbar; % density at radius r / mean density  within Sr
iratio=inertia(n)/mass(n)/R^2;% Inertia to mass ratio

% Choice of GM
% The solutions depend on density only via G * density. Thus you can decide
% wether M, d, dbar, etc are the one of your model or if GM is to be the
% one of the actual Earth. In the second case we renormalize density etc :
% this is only to put coherence between our values of GM, M and G and has
% no influence upon the calculated figure.
if iGM==1
GM=G*mass(n); % Case where GM is G * Mass of your model.
end
if iGM==0
GM=GM0;       % Case where GM is the actual GM of the Earth.
renorm=GM/(G*mass(n));
mass=mass*renorm; % Renormalisations only for plot and consistency
d=d*renorm;
dbar=dbar*renorm;
iratiomodel=iratio/renorm; % iratio with the good M
end

% Over global quantities
g=GM/R^2*dbar.*r./dbar(n)/R; % Gravity at radius r : not needed, only for plot.
m=om^2*R^3/GM;   % Adimensional centifugal acceleration at surface. 
                 % The 'first' or 'second' order refers to that small
                 % parameter
% m=0.00345039;   % Value used by Carlo Denis, 1989
% m=0.003449236;  % Value estimated with PREM parameters.
% m=0.00344980146;% Value infered from Nakiboglu 1979
mr=m*dbar(n)./dbar; % Vector of m(r)=omega^2 r/g(r)

% Reference ellipsoid parameters (WGS84 ellipsoid)
a84=6378137;         % major semi-axis
b84=6356752.3142;    % minor semi-axis	
f84=1/298.257223563; % flattening (a-b)/a
% Best fitting ellipsoid parameters
% At first order f=3J2/2+m/2 thus, due to a change of radius:
df84=3*(R-6371000.)/R*m/2;
f_ell=f84+df84;
a_ell=R/(1-f_ell)^(1./3.);
b_ell=a_ell*(1-f_ell);
% (a change of a_ell of about 0.230/6371=1/3e4 changes f of less than 1/1e4
% thus in these formulae we can keep the same f when changing the radius R)

% Display some global constants
disp(' ')
disp(sprintf('%s %11.2f','Model radius R (m) =',R))
disp(sprintf('%s %11.2f','Earth radius R (m) =',R0))
disp(sprintf('%s %1.8g' ,'Model mass M (kg)  =',mass(n)))
disp(sprintf('%s %1.8g' ,'Earth mass M (kg)  =',GM0/G))
disp(sprintf('%s %1.8g' ,'Model GM (s.i)     =',GM))
disp(sprintf('%s %1.8g' ,'Earth GM (s.i)     =',GM0))
disp(sprintf('%s %1.8f','Model inertia ratio I/MR^2 =',iratio))
disp(sprintf('%s %1.8f','Earth inertia ratio I/MR^2 =',iratio0))
%disp(sprintf('%s %1.8f','Surface gravity (m.^s-2)   =', g(n)))
disp(sprintf('%s %1.8f','Model m = om^2*R^3/GM      =',m))
disp(sprintf('%s %1.8f','Earth m = om^2*R^3/GM      =',m0))
disp(sprintf('%s %1.8f','Best fitting ellipsoid 1/flattening =',1/f_ell))
disp(sprintf('%s %1.8f','Best fitting ellipsoid major semi-axis =',a_ell))
disp(sprintf('%s %1.8f','Best fitting ellipsoid minor semi-axis =',b_ell))
disp(' ')
% Nakiboglu's values
f_naki=1/299.627;
a_naki=6371000./(1-f_naki)^(1./3.);
b_naki=6371000.*(1-f_naki)^(2./3.);
disp(sprintf('%s %1.8f','Surface minor semi-axis difference (meters), b = ',b84-a84-(b_naki-a_naki)))


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   First order integration                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Integrates the original (first order) 'Clairaut equation'.

disp('First order integration')
% For 1st and 2nd order integration, shears tables with functions func*.m
global ppgam; % spline of gamma
global heterogeneous; % Shears the homogenity of the system to integrate
                      % Heterogeneous=0 : integrates the homogeneous system
                      % Heterogeneous=1 : integrates the heterogeneous system

heterogeneous=0;  % Integrates the homogeneous system
y0=[1,0]; % Starting vector (initial condition at the center)
F=[];     % Vector to integrate, first column = f2, 
          % second column = r partial_r f2

% Integration of the system. That must be done layer by layer because :
% 1. radii at discontinuities appear twice,
% 2. possible extrapolation with ode45 must use values of gamma (and 
%    of y1 and y2 at second order) in the same layer.
% Vectors rspan and gamspan are restrictions in the layer of r and gamma.

for j=1:J
rspan=ri(nstart(j):nend(j));
gamspan=gamma(nstart(j):nend(j));
ppgam=interp1(rspan,gamspan,'spline','pp');
[T,y]=ode45(@degree2,rspan,y0,options);
if length(rspan)==2 % If two points, ode45 returns intermediate points
    y=[y(1,:);y(length(y),:)];
end
F=[F;y];          % Solution in layer is appended to the whole solution
y0=y(length(y),:);% Starting vector in next layer (F is continuous)
end

% Applies the surface condition
F=-F*5*m/3/(2*F(n,1)+F(n,2)); % F is thus the solution at first order

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Second order integration                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Integrates the generalized (correct to second order) 'Clairaut equations'.
% Adds general solution (homogeneous system) and particular solution
% (heterogenenous system).

disp('Second order integration')
disp(' ')
global ppgam pps2 pps4
% Initializes both solutions for both degrees
F2=[]; F4=[];     % Degree 2 & 4, heterogeneous system 
F2G=[];F4G=[];    % Same for homogeneous system

% Second members of heterogeneous differential systems * r for
% degree 2 (s2) and 4 (s4).
y1=F(:,1);y2=F(:,2);
y1=y1+F2saved(:,1);y2=y2+F2saved(:,2); % Just in case of iteration
s2=2/7*(2*y2.*(y2+9*y1)-9*gamma.*y2.*(y2+2*y1))+4*mr.*(1-gamma).*(y2+y1);
s4=18/35*(2*y2.*(y2+2*y1)-3*gamma.*(3*y2.^2+6*y2.*y1+7*y1.^2));

% Linear combinations at surface for the limit conditions
CL2=2/3*m*(5*y1(n)+y2(n))+1/7*(2*y2(n)^2+6*y1(n)*y2(n)+12*y1(n)^2);
% Nakiboglu's condition with error :
% CL2=2/3*m*(5*y1(n)+y2(n))+1/7*(2*y2(n)^2+y1(n)*y2(n)+12*y1(n)^2); 
CL4=18/35*(y2(n)^2+5*y1(n)*y2(n)+6*y1(n)^2);

% Starting vectors at r=r0
% Particular solution (heterogeneous system)
y02=[0,0]; % Degree 2
y04=27/35*y1(1)^2*[1,0]; % Degree 4
% General solution (homogeneous system)
y02G=[1,0]; % Degree 2
y04G=[1,2]; % Degree 4

% Integration of the systems for degrees 2 and 4.
for j=1:J
rspan=ri(nstart(j):nend(j));
gamspan=gamma(nstart(j):nend(j));
s2span=s2(nstart(j):nend(j));
s4span=s4(nstart(j):nend(j));
ppgam=interp1(rspan,gamspan,'spline','pp');
pps2=interp1(rspan,s2span,'spline','pp');
pps4=interp1(rspan,s4span,'spline','pp');
heterogeneous=1; % Heterogeneous system
[T2,y2]=ode45(@degree2,rspan,y02,options);
[T4,y4]=ode45(@degree4,rspan,y04,options);
if length(rspan)==2 % If two points, ode45 returns intermediate points
    y2=[y2(1,:);y2(length(y2),:)];
    y4=[y4(1,:);y4(length(y4),:)];
end
F2=[F2;y2];
F4=[F4;y4];
y02=y2(length(y2),:);
y04=y4(length(y4),:);
heterogeneous=0; % Homogeneous system
[T2,y2]=ode45(@degree2,rspan,y02G,options);
[T4,y4]=ode45(@degree4,rspan,y04G,options);
if length(rspan)==2 % If two points, ode45 returns intermediate points
    y2=[y2(1,:);y2(length(y2),:)];
    y4=[y4(1,:);y4(length(y4),:)];
end
F2G=[F2G;y2];
F4G=[F4G;y4];
y02G=y2(length(y2),:);
y04G=y4(length(y4),:);
end

% Normalises F2G and F4G
F2G=F2G/(2*F2G(n,1)+F2G(n,2));
F4G=F4G/(4*F4G(n,1)+F4G(n,2));
F2=F2-(2*F2(n,1)+F2(n,2))*F2G;
F4=F4-(4*F4(n,1)+F4(n,2))*F4G;
F2G=F2G*CL2;
F4G=F4G*CL4;

% Figures of these functions
if ifigure == 1
figure;title('F');hold on;plot(r,F(:,1),'r')
figure;title('F(2)');hold on;plot(r,F(:,2),'r')

figure;title('F2 et -F2G');hold on;plot(r,F2(:,1),'r',r,-F2G(:,1),'k')
figure;title('F4 et -F4G');hold on;plot(r,F4(:,1),'r',r,-F4G(:,1),'k')

figure;title('F2(2)');hold on;plot(r,F2(:,2),'r')
figure;title('F4(2)');hold on;plot(r,F4(:,2),'r')
figure;title('F2G(2)');hold on;plot(r,F2G(:,2),'k')
figure;title('F4G(2)');hold on;plot(r,F4G(:,2),'k')
end

% Combine particular and general solution 
% to obtain F2 and F4 the degree 2 and 4 solutions correct to second order
F2=F2+F2G; 
F4=F4+F4G;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Control of limit conditions                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

test1=abs(F2(1,2)/F2(1,1)); % At the center, degree 2
test3=abs(F4(1,2)/F4(1,1)); % At the center, degree 4
test2=abs((2*F2(n,1)+F2(n,2))/CL2-1); % Surface, degree 2
test4=abs((4*F4(n,1)+F4(n,2))/CL4-1); % Surface, degree 2
disp(' ')
disp('Relative error in the four 2nd order limit conditions')
disp(sprintf(' %3.3g %3.3g %3.3g %3.3g',test1,test2,test3,test4))
if max([test1,test2,test3,test4])>1e-6
disp('*********                   Break:                       *****')
disp('********* limit conditions not satisfied at required level *****')
stop
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Derived results and plots                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

F2saved=F2; % Save of F2 (second order only)

% F2 is now the sum of first and second order :
F2=F+F2;

eta=F(:,2)./F(:,1);   % r*Logarithmic derivative of f2 at first order
eta2=F2(:,2)./F2(:,1);% r*Logarithmic derivative of f2 correct to second order
eta4=F4(:,2)./F4(:,1);% r*Logarithmic derivative of f4 correct to second order

f=F(n,1); % f at surface (first order)
f2=F2(n,1); % f2 at surface (second order)
f4=F4(n,1); % f4 at surface

% Flattening (from Nakiboglu, eq. 7)
e=-3*F(:,1)/2;  % first order
e2=-3*F2(:,1)/2-5*F4(:,1)/8-3*F2(:,1).^2/8; % second order
% Gravitationnal potential coefficients (Nakiboglu, eq. 7)
J2first=-f-m/3; % first order
J2=-f2-m/3-11*f2^2/7-m*f2/7; % second order
J4=-f4-36*f2^2/35-6*m*f2/7;
% Fluid Love number
kf=5/(2+eta(n))-1; % same as kf=3*J2first/m 
% These coefficients are associated to the following major semi-axis a 
% of the surface:
a=R*(1-f2/2+3*f4/8);
% Renormalize in such a way that they are associated with the scale length
% ae defined in first lines of programm :
J2first=J2first*(a/ae)^2;
J2=J2*(a/ae)^2;
J4=J4*(a/ae)^4;

% Minor semi-axis at surface :
b=R*(1+f2+f4);
% At the CMB :
ncmb=nend(2);
acmb=r(ncmb)*(1-F2(ncmb,1)/2+3*F4(ncmb,1)/8);  %major semi-axis
bcmb=r(ncmb)*(1+F2(ncmb,1)+F4(ncmb,1));        %minor semi-axis
% Flatenings, with the definition (a-b)/a of the Geodetic Reference System
esurface=(a-b)/a;
ecmb=(acmb-bcmb)/acmb;

% Display several numbers
disp(' ')
disp(sprintf('%s %1.8f','Surface f2 *1e3, 1st order =',f*1e3))
disp(sprintf('%s %1.8f','    "      "     2nd order =',f2*1e3))
disp(sprintf('%s %1.8f','    "   f4 *1e6, 2nd order =',f4*1e6))
disp(sprintf('%s %1.8f','CMB     f2 *1e3, 1st order =',F(ncmb,1)*1e3))
disp(sprintf('%s %1.8f','    "      "     2nd order =',F2(ncmb,1)*1e3))
disp(sprintf('%s %1.8f','    "   f4 *1e6, 2nd order =',F4(ncmb,1)*1e6))
disp(' ')
%disp(sprintf('%s %1.8f','Fluid Love Number, 1st order =',kf ))
disp(sprintf('%s %1.8f','Surface major semi-axis (meters), a = ',a))
disp(sprintf('%s %1.8f','Surface minor semi-axis (meters), b = ',b))
disp(sprintf('%s %1.8f','Surface minor semi-axis difference (meters), b = ',b_ell-a_ell-(b-a)))
disp(sprintf('%s %1.8f','CMB     major semi-axis (meters), acmb = ',acmb))
disp(sprintf('%s %1.8f','CMB     minor semi-axis (meters), bcmb = ',bcmb))
%disp(sprintf('%s %1.8f','CMB     minor semi-axis difference (meters), b = ',b-bcmb))
disp(sprintf('%s %1.8f','Surface inverse of flattening =',1./esurface ))
disp(sprintf('%s %1.8f','CMB     inverse of flattening =',1./ecmb ))
disp(' ')
disp(sprintf('%s %1.8f','J2first *1e6, 1st order  = ',J2first*1e6))
disp(sprintf('%s %1.8f','J2      *1e6, 2nd order  = ',J2*1e6))
disp(sprintf('%s %1.8f','J4      *1e6, 2nd order  = ',J4*1e6))
disp(sprintf('%s %1.8f','(J2-J2first)  *1e6       = ',(J2-J2first)*1e6))


% Some figures
if ifigure == 1
figure;title('1/e (k) & 1/e2 (r)');hold on;
plot(r,1./e,'k',r,1./e2,'r')
figure;title('r*Log derivative of e (k), e2(r) and e4 (b)');hold on;
plot(r,eta,'k',r,eta2,'r',r,eta4,'b')
figure;title('f2*1e4 (k) f4*1e7 (r)');hold on;
plot(r,-F2(:,1)*1e4,'k',r,F4(:,1)*1e7,'r')
figure;title('1000*m (r) & gamma (k)');hold on;
plot(r,1000*mr,'r',r,gamma,'k')
figure;title('density in g/cm3 (r) & gravity in m/s-2 (k)');hold on;
plot(r,d/1000,'r',r,g,'k')
end

% Write 1000*f(r), 1000*f2(r) and 1e6*f4(r) in a file
fid=fopen('f.dat','w+');  
for i=1:n
fprintf(fid,'%10.2f %1.9f %1.9f %1.9f \n',r(i),1e3*F(i,1),1e3*F2(i,1),1e6*F4(i,1));
end
fclose(fid);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   Radau results and correction for inertia      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compare to J2 results found with Radau theorem at first order,
% with I/MR^2 of the model used and with the actual one of the Earth. 
% We propose k and J2 that correspond to the I/MR^2 of the real Earth, 
% by applying a correction that is the difference between the the Radau 
% value with I/MR^2 of the model and the Radau value with I/MR^2 of the Earth.

% Intermediate quantity k : Love number
k_model=5/(25/4*(1-3/2*iratio)^2+1)-1;
k_earth=5/(25/4*(1-3/2*iratio0)^2+1)-1;
% Proposed dynamical flattening J2
J2R_model=k_model*m/3*(a/ae)^2;
J2R_earth=k_earth*m/3*(a/ae)^2;
disp(sprintf('%s %1.8f','J2 Radau*1e6 of model    = ',J2R_model*1e6))
disp(sprintf('%s %1.8f','J2 Radau*1e6 of Earth    = ',J2R_earth*1e6))
J2_corrected=J2+J2R_earth-J2R_model;
% (One could also propose : J2_corrected=J2R_earth+J2-J2first);
disp(sprintf('%s %1.8f %s','J2 best *1e6,            = ',J2_corrected*1e6,' <- Proposed estimation (2nd order'))
disp(sprintf('%s','                                              with best estimations of'))
disp(sprintf('%s','                                              GM, R, I/MR^2 and Omega)'))
disp(' ')

% Proposed Love number k
disp(sprintf('%s %1.8f','k of model          = ',kf))
disp(sprintf('%s %1.8f','k Radau of model    = ',k_model))
disp(sprintf('%s %1.8f','k Radau of Earth    = ',k_earth))
k_corrected=kf+k_earth-k_model;
disp(sprintf('%s %1.8f %s','k best              = ',k_corrected,' <- Proposed estimation' ))
disp(' ')

% C20
J2obs=1.0826046e-3;
J4obs=-1.620e-6;
J2Naki=1.072701e-3;
J4Naki=-2.992e-6;
disp(sprintf('%s %1.8f','C20*1e6 obs    = ',-J2obs/sqrt(5)*1e6))
disp(sprintf('%s %1.8f','C20*1e6 hydrostatic moi   = ',-J2_corrected/sqrt(5)*1e6))
disp(sprintf('%s %1.8f','C20*1e6 hydrostatic Naki   = ',-J2Naki/sqrt(5)*1e6))
disp(sprintf('%s %1.8f','C20*1e6 obs-hydrostatic moi   = ',-(J2obs-J2_corrected)/sqrt(5)*1e6))
disp(sprintf('%s %1.8f','C20*1e6 obs-hydrostatic Naki   = ',-(J2obs-J2Naki)/sqrt(5)*1e6))
disp('')
disp(sprintf('%s %1.8f','C40*1e6 obs    = ',-J4obs/3*1e6))
disp(sprintf('%s %1.8f','C40*1e6 hydrostatic moi   = ',-J4/3*1e6))
disp(sprintf('%s %1.8f','C40*1e6 hydrostatic Naki   = ',-J4Naki/3*1e6))
disp(sprintf('%s %1.8f','C40*1e6 obs-hydrostatic moi   = ',-(J4obs-J4)/3*1e6))
disp(sprintf('%s %1.8f','C40*1e6 obs-hydrostatic Naki   = ',-(J4obs-J4Naki)/3*1e6))

disp('End of hydrostatic.m')

